/*
 * @(#)AMActionFigure.java  1.0  2006-12-18
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.am.diagram;

import ch.hslu.cm.Diagram;
import ch.hslu.cm.DiagramFigure;
import ch.hslu.cm.am.model.AMAction;
import java.io.IOException;
import ch.hslu.cm.simulation.*;
import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;
import java.util.*;
import javax.swing.undo.*;
import java.beans.*;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.connector.Connector;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.handle.Handle;
import org.jhotdraw.draw.handle.MoveHandle;
import org.jhotdraw.draw.locator.RelativeDecoratorLocator;
import org.jhotdraw.geom.*;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.*;

/**
 * Represents the FIGURE (graphical, visual representation) of an Action element
 * in an Activity Diagram.
 * <p>
 * An action may have sets of incoming and outgoing activity edges that specify
 * control flow to other nodes. An action will not begin execution until all of
 * its input conditions are satisfied. The completion of the execution of an
 * action may enable the execution of a set of successor nodes that take their
 * inputs from the outputs of the action.
 * <p>
 * Actions are notated as round-cornered rectangles. The name of the action or
 * other description of it may appear in the symbol.
 * <p>
 * For further information, consult the official UML
 * <a href="http://www.omg.org/docs/formal/05-07-04.pdf">online resource</a>
 * of the Object Management Group (OMG):<br>
 * Unified Modeling Language: Superstructure, version 2.0, formal/05-07-04,
 * page 301 et sqq.
 *
 * @author Florian Padrun
 * @version 1.0 2006-12-18 Created.
 */
public class AMActionFigure extends TextFigure
        implements DiagramFigure, PropertyChangeListener {

    /** Figure model. */
    private AMAction model;

    /**
     * This adapter is used, to connect a TextFigure with the name of
     * the AMAction model.
     */
    private static class NameAdapter extends FigureAdapter {

        private AMActionFigure target;

        public NameAdapter(AMActionFigure target) {
            this.target = target;
        }

        @Override
        public void attributeChanged(final FigureEvent e) {
            if (e.getAttribute().equals(TEXT)) {
                target.model.setName((String) e.getNewValue());

                target.fireUndoableEditHappened(new AbstractUndoableEdit() {

                    @Override
                    public String getPresentationName() {
                        ResourceBundleUtil labels = ResourceBundleUtil.getBundle(
                                "ch.hslu.cm.am.Labels", Locale.getDefault());
                        return labels.getString("name");
                    }

                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        target.model.setName((String) e.getOldValue());
                    }

                    @Override
                    public void redo() throws CannotUndoException {
                        super.redo();
                        target.model.setName((String) e.getNewValue());
                    }
                });
            }
        }
    }
    /** Name of the figure */
    private NameAdapter nameAdapter;

    /** Creates a new instance. */
    public AMActionFigure() {
        RoundRectangleFigure decorator = new RoundRectangleFigure();
        STROKE_PLACEMENT.set(decorator, AttributeKeys.StrokePlacement.OUTSIDE);
        FILL_UNDER_STROKE.set(decorator, AttributeKeys.Underfill.NONE);
        decorator.setArc(20, 20);
        setDecorator(decorator);
        DECORATOR_INSETS.set(this, new Insets2D.Double(7, 8, 7, 8));

        FILL_COLOR.set(this, ActivityDiagram.FILL_COLOR);
        STROKE_COLOR.set(this, ActivityDiagram.STROKE_COLOR);
        TEXT_COLOR.set(this, ActivityDiagram.TEXT_COLOR);
        STROKE_WIDTH.set(this, ActivityDiagram.STROKE_WIDTH);

        setEditable(true);
        FONT_BOLD.set(this, true);
        addFigureListener(nameAdapter = new NameAdapter(this));

        setModel(createAMAction());

        setAttributeEnabled(DECORATOR_INSETS, false);
        setAttributeEnabled(STROKE_TYPE, false);
        setAttributeEnabled(STROKE_PLACEMENT, false);
        setAttributeEnabled(FILL_UNDER_STROKE, false);
        setAttributeEnabled(FONT_BOLD, false);
        setAttributeEnabled(FONT_ITALIC, false);
        setAttributeEnabled(FONT_UNDERLINE, false);

        setAttributeEnabled(STROKE_WIDTH, false);
        setAttributeEnabled(STROKE_COLOR, false);
        setAttributeEnabled(STROKE_DASHES, false);
        setAttributeEnabled(TEXT_COLOR, false);
    }

    /**
     * Draws the text of the figure.
     * @param g Graphics2D
     */
    @Override
    protected void drawText(java.awt.Graphics2D g) {
        if (getText() != null || isEditable()) {
            TextLayout layout = getTextLayout();
            layout.draw(g, (float) origin.x,
                    (float) (origin.y + layout.getAscent()));
        }
    }

    /**
     * Draws the connector of the figure.
     * @param g Graphics2D
     */
    protected void drawConnectors(Graphics2D g) {
        AMActionConnector connector = new AMActionConnector(this);
        connector.draw(g);
    }

    /**
     * Sets an attribute of the figure without firing events.
     * @param key AttributedKey
     * @param newValue Object
     */
    @Override
    public void set(AttributeKey key, Object newValue) {
        super.set(key, newValue);
        if (getDecorator() != null) {
            getDecorator().set(key, newValue);
        }
    }

    /**
     * Creates and returns handles used to manipulate the figure.
     * @param detailLevel int
     * @return handles
     */
    @Override
    public Collection<Handle> createHandles(int detailLevel) {
        LinkedList<Handle> handles = new LinkedList<Handle>();
        if (detailLevel == 0) {
            handles.add(new MoveHandle(this,
                    RelativeDecoratorLocator.northEast()));
            handles.add(new MoveHandle(this,
                    RelativeDecoratorLocator.northWest()));
            handles.add(new MoveHandle(this,
                    RelativeDecoratorLocator.southEast()));
            handles.add(new MoveHandle(this,
                    RelativeDecoratorLocator.southWest()));
        }
        return handles;
    }

    /**
     * Returns the number of connections.
     * @return connectionCount
     */
    public int getConnectionCount() {
        return getModel().getRelationships().size();
    }

    /**
     * Returns the index in this list of the first occurrence of the specified
     * element, or -1 if this list does not contain this element.
     * @param f DiagramFigure
     * @return connectionIndex
     */
    public int getConnectionIndex(DiagramFigure f) {
        return getModel().getRelationships().indexOf(f.getModel());
    }

    /**
     * Gets a connector for this figure at the given location.
     * A figure can have different connectors at different locations.
     * @param p Point2D.Double
     * @param prototype ConnectionFigure
     */
    @Override
    public Connector findConnector(Point2D.Double p,
            ConnectionFigure prototype) {
        if (contains(p)) {
            return new AMActionConnector(this);
        } else {
            return null;
        }
    }

    /**
     * Gets a compatible connector.
     * If the provided connector is part of this figure, return the connector.
     * If the provided connector is part of another figure, return a connector
     * with the same semantics for this figure.
     * Return null, if no compatible connector is available.
     * @param c Connector
     * @param isStart boolean
     */
    @Override
    public Connector findCompatibleConnector(Connector c,
            boolean isStart) {
        if (c.getOwner() == this) {
            return c;
        }
        return new AMActionConnector(this);
    }

    /**
     * Creates and returns a model object of the figure.
     * @return action
     */
    protected AMAction createAMAction() {
        AMAction action = new AMAction();
        return action;
    }

    /**
     * Sets the model of the figure.
     * @param m AMAction
     */
    public void setModel(AMAction m) {
        willChange();

        if (model != null) {
            model.removePropertyChangeListener(this);
        }
        model = m;

        if (model != null) {
            model.addPropertyChangeListener(this);
            setText(model.getName());
        }
        changed();
    }

    /**
     * Gets the model of the figure.
     * @return model
     */
    @Override
    public AMAction getModel() {
        return model;
    }

    /**
     * Gets the container for diagram figures.
     * @return diagram
     */
    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    /**
     * Gets the simulation (collection of elements) of the diagram.
     * @return simulation
     */
    private Simulation getSimulation() {
        return getDiagram().getSimulation();
    }

    /**
     * Informs the figure, that it has been added to the specified drawing.
     * The figure must inform all FigureListeners that it has been added.
     * @param drawing Drawing
     */
    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }

    /**
     * Informs the figure, that it has been removed from the specified drawing.
     * The figure must inform all FigureListeners that it has been removed.
     * @param drawing Drawing
     */
    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    /**
     * Creates and returns a copy of the figure object.
     * @return that
     */
    @Override
    public AMActionFigure clone() {
        AMActionFigure that = (AMActionFigure) super.clone();
        that.nameAdapter = new NameAdapter(that);
        that.addFigureListener(that.nameAdapter);
        that.setModel((AMAction) this.model.clone());
        return that;
    }

    /**
     * Reads the persistent figure object.
     * @param in DOMInput
     */
    @Override
    public void read(DOMInput in) throws IOException {
        double x = in.getAttribute("x", 0d);
        double y = in.getAttribute("y", 0d);
        double w = in.getAttribute("w", 0d);
        double h = in.getAttribute("h", 0d);
        setBounds(new Point2D.Double(x, y), new Point2D.Double(x + w, y + h));
        readAttributes(in);
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((AMAction) in.readObject(0));
        in.closeElement();
    }

    /**
     * Writes the figure object in a persisten state.
     * @param out DOMOutput
     */
    @Override
    public void write(DOMOutput out) throws IOException {
        Rectangle2D.Double r = getBounds();
        out.addAttribute("x", r.x);
        out.addAttribute("y", r.y);
        writeAttributes(out);
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
    }

    /**
     * Gets the layer of the figure in the diagram.
     * @return ACTIVITY_LAYER
     */
    @Override
    public int getLayer() {
        return ActivityDiagram.ACTIVITY_LAYER;
    }

    /**
     * This method gets called when a bound property is changed.
     * It informs that a Figure is about to change its shape.
     * It informs that a Figure changed its shape.
     * @param evt PropertyChangeEvent
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        willChange();
        changed();
    }
}
